/*
  To Folder
    あかつきみさき

  このスクリプトについて
    フォルダーを一覧で表示し,選択したアイテム・レイヤーソースをリストでクリックしたフォルダーに格納するスクリプト.

  使用方法
    1.ファイル→スクリプト→スクリプトファイルの実行から実行.
    2.ScriptUIフォルダに入れてから,ウィンドウより実行.

  動作環境
    Adobe After Effects 2024以上

  バージョン情報
    2026/01/05 Ver 2.0.0 Update
      リストをツリービューに変更.
      リストをクリックで移動するように変更.
      レイヤーを選択している時、レイヤーのソースアイテムに対して処理を実行するようにした.
      Undo処理のテキストに格納先のフォルダー名を表示するようにした.
      内部処理の改善.

    2016/07/24 Ver 1.1.0 Update
      ダブルクリックで移動するように変更.
      Undo処理の追加.

    2016/07/24 Ver 1.0.1 Update
      マウス動作時リストが読み込まれ続ける問題の修正.

    2016/07/24 Ver 1.0.0 Release
 */

(function () {
  var data = {
    scriptName: "To Folder",
    scriptVersion: "2.0.0"
  };

  function buildUI(thisObj) {
    var tfWin;
    if (thisObj instanceof Panel) {
      tfWin = thisObj;
    } else {
      tfWin = new Window("palette", data.scriptName + " - " + data.scriptVersion, undefined, {
        resizeable: true
      });
    }

    tfWin.orientation = "column";
    tfWin.alignment = [ScriptUI.Alignment.FILL, ScriptUI.Alignment.FILL];
    tfWin.margins = [0, 0, 0, 0];
    tfWin.spacing = 1;

    var headerMenu = tfWin.add("group");
    headerMenu.alignment = [ScriptUI.Alignment.FILL, ScriptUI.Alignment.TOP];
    headerMenu.spacing = 0;
    headerMenu.margins = [0, 0, 0, 0];

    var reloadBtn = headerMenu.add("button");
    reloadBtn.text = "Reload";
    reloadBtn.alignment = [ScriptUI.Alignment.FILL, ScriptUI.Alignment.FILL];
    reloadBtn.helpTip = "";

    var helpBtn = headerMenu.add("button");
    helpBtn.text = "?";
    helpBtn.alignment = [ScriptUI.Alignment.RIGHT, ScriptUI.Alignment.TOP];
    helpBtn.minimumSize.width = helpBtn.maximumSize.width = 25;
    helpBtn.helpTip = "Setting";

    var folderList = tfWin.add("treeview");
    folderList.alignment = [ScriptUI.Alignment.FILL, ScriptUI.Alignment.FILL];
    folderList.spacing = 0;
    folderList.margins = [0, 0, 0, 0];

    reloadBtn.onClick = function () {
      folderList.removeAll();
      var allItems = app.project.items;
      var folderMap = {};
      
      var rootNode = folderList.add("node", app.project.rootFolder.name);
      rootNode.folder = app.project.rootFolder;
      folderMap[0] = rootNode;
      
      for (var i = 1; i <= allItems.length; i++) {
        if (!(allItems[i] instanceof FolderItem)) {
          continue;
        }
        
        var parentFolder = allItems[i].parentFolder;
        var parentNode = folderMap[parentFolder.id];
        if (!parentNode) {
          parentNode = rootNode;
        }
        
        var node = parentNode.add("node", allItems[i].name);
        node.folder = allItems[i];
        folderMap[allItems[i].id] = node;
      }
      
      for (var folderId in folderMap) {
        folderMap[folderId].expanded = true;
      }
    };

    folderList.onChange = function () {
      if (this.selection == null) {
        return;
      }

      var actComp = app.project.activeItem;
      app.beginUndoGroup("選択アイテムを「" + this.selection.folder.name + "」に格納 - " + data.scriptName);

      // レイヤーが選択されている場合
      if (actComp instanceof CompItem && actComp.selectedLayers.length > 0) {
        var selectedLayers = actComp.selectedLayers;
        for (var i = 0; i < selectedLayers.length; i++) {
          var sourceItem = selectedLayers[i].source;
          if (sourceItem != null && this.selection.folder != null) {
            sourceItem.parentFolder = this.selection.folder;
          }
        }
      } else {
        // レイヤーが選択されていない場合、プロジェクトアイテムに処理
        var selectionItems = app.project.selection;
        for (var i = 0; i < selectionItems.length; i++) {
          if (this.selection.folder == selectionItems[i].parentFolder) {
            continue;
          }

          if (this.selection.folder != null) {
            selectionItems[i].parentFolder = this.selection.folder;
          }
        }
      }

      app.endUndoGroup();
      this.selection = null;

      reloadBtn.onClick();
    };

    helpBtn.onClick = function () {
      alert(
        data.scriptName + " - " + data.scriptVersion + "\n" +
        localize({
          ja: "あかつきみさき (SUNRISE MOON)",
          en: "Misaki_Akatsuki (SUNRISE MOON)"
        }) + "\n\n" +
        localize({
          ja: "フォルダーを一覧で表示し,選択したアイテム・レイヤーソースをリストでクリックしたフォルダーに格納するスクリプト.",
          en: "Set parent folder"
        }) + "\n",
        data.scriptName
      );
    };

    reloadBtn.onClick();

    tfWin.layout.layout(true);
    tfWin.onResizing = tfWin.onResize = function () {
      this.layout.resize();
    };

    return tfWin;
  }

  var lbPanl = buildUI(this);
  if (!(lbPanl instanceof Panel)) {
    lbPanl.center();
    lbPanl.show();
  }

  return 0;
}).call(this);
